'
' loads a list of "extreme penalty" IPs from ASSP into the
' windows firewall to lock them out from the local box and
' avoid "port rattling"; the IPs will be cleaned up upon the
' next ASSP extreme IPs export (if they aren't included in
' the extreme IP list)
'

Option Explicit

' file I/O
Const ForReading = 1, ForWriting = 2, ForAppending = 8

' FW: Direction
Const NET_FW_RULE_DIR_IN = 1
Const NET_FW_RULE_DIR_OUT = 2

' FW: Action
Const NET_FW_ACTION_BLOCK = 0
Const NET_FW_ACTION_ALLOW = 1

' FW: misc
Const FW_IP_LIST_FILE = "exportedextreme.txt"
Const FW_RULE_BASE_NAME = "ASSP extreme #"
Const FW_RULE_DESCRIPTION = "ASSP extreme IPs blocking"
Const FW_MAX_IP_PER_RULE = 100

' globals
Dim arg, fso
Dim gsBasePath, gsBaseName
Dim gsLogFile, gsIniFile
Dim gdicConfig

' firewall stuff
Dim goFwPolicy, goFwRules, goFwCurrProf
Dim gnMaxIpPerRule, gsRulePrefix, gsRuleDesc, gbRuleEnabled

' starts main and returns errorlevel
Set arg = WScript.Arguments
Set fso = CreateObject("Scripting.FileSystemObject")
WScript.Quit main(arg.Count, arg)

' main entry point
Function main(argc, argv)
  Dim sRuleFile, iRule

  ' init
  If Not initScript() Then  
    printf "Error while initializing script."
    main = 1
    Exit Function
  End If  

  printf "Processing ASSP firewall rules..."

  ' if we've an arg, it's the "ini" file name
  If argc > 0 Then
    gsIniFile = Trim(argv(0))
  End If
  
  ' loads config file
  If Not loadConfig(gsIniFile) Then
    printf "Error while loading configuration from file."
  '  main = 2
  '  Exit Function
  End If  
  
  ' setup config/default values
  sRuleFile = getConfValue("IPListFile", FW_IP_LIST_FILE)
  gnMaxIpPerRule = CInt(getConfValue("MaxIpPerRule", FW_MAX_IP_PER_RULE))
  gsRulePrefix = getConfValue("RuleBaseName", FW_RULE_BASE_NAME)
  gsRuleDesc = getConfValue("RuleDesc", FW_RULE_DESCRIPTION)
  gbRuleEnabled = CBool(getConfValue("EnableRules", True))

  printf "Filtered IP file..........: " & sRuleFile
  printf "Max IP(s) in each rule....: " & gnMaxIpPerRule
  printf "Rules prefix..............: " & gsRulePrefix
  printf "Rules description.........: " & gsRuleDesc
  printf "Rules enabled.............: " & gbRuleEnabled

  ' initializes WFW interfaces
  If Not initFW() Then
    printf "Error while initializing firewall interfaces."
    main = 3
    Exit Function
  End If
  
  ' load IPs from file and create rules
  If Not loadRules(sRuleFile) Then
    printf "Error while loading FW rules, check log file."
    main = 4
    Exit Function
  End If
 
  ' all ok
  printf "Processing completed."
  main = 0
End Function

' ===========================================================================
' CUSTOM
' ===========================================================================

' setup the WFW interfaces
Function initFW()
  
  On Error Resume Next
  initFW = False
  Err.Clear
  
  ' init
  printf "Initializing Windows Firewall interface ..."
  Set goFwPolicy = CreateObject("HNetCfg.FwPolicy2")
  If Err.Number <> 0 Then
    perror "Create FW policy object"
    Exit Function
  End If  
  
  ' get a reference to rules
  Set goFwRules = goFwPolicy.Rules
  If Err.Number <> 0 Then
    perror "Init FW policy rules"
    Exit Function
  End If  
  
  ' get a reference to profile
  goFwCurrProf = goFwPolicy.CurrentProfileTypes
  If Err.Number <> 0 Then
    perror "Init FW current profile"
    Exit Function
  End If  

  ' all ok
  initFW = True
End Function

' loads WFW rules from file
Function loadRules(sFileName)
  Dim sBuff, vaRec, iRec
  Dim fp, sIP, sAddr
  Dim nRule, nAddr
  Dim nErr
  
  On Error Resume Next
  loadRules = False
  Err.Clear
  
  ' open and load the IP list file
  printf "Loading rules from file " & sFileName & " ..."
  Set fp = fso.OpenTextFile(sFileName, ForReading, False)
  If Err.Number <> 0 Then
    perror "opening rule file " & sFileName
    Exit Function
  End If
  sBuff = ""
  If Not fp.AtEndOfStream Then
    sBuff = fp.ReadAll
  End If
  fp.Close
  If Len(sBuff) < 1 Then
    printf "Error: rule file is empty or cannot be read."
    Exit Function
  End If
  sBuff = Replace(sBuff, vbCrLf, vbLf)
  sBuff = Replace(sBuff, vbCr, vbLf)
  vaRec = Split(sBuff, vbLf)
  
  ' load the rules
  printf "Loading " & UBound(vaRec) + 1 & " IP(s) into firewall rules..."
  sAddr = ""
  nRule = 0
  nAddr = 0
  For iRec = LBound(vaRec) To UBound(vaRec)
    sIP = Trim(vaRec(iRec))
    If Len(sIP) > 0 Then
      If nAddr < gnMaxIpPerRule Then
        sAddr = sAddr & "," & sIP
        nAddr = nAddr + 1
      Else
        nRule = nRule + 1
        If Not addRule(Mid(sAddr, 2), nRule) Then
          nErr = nErr + 1
        End If
        sAddr = ""
        nAddr = 0
      End If
    End If
  Next
  
  ' last rule
  If Len(sAddr) > 0 Then
    nRule = nRule + 1
    If Not addRule(Mid(sAddr, 2), nRule) Then
      nErr = nErr + 1
    End If
  End If

  ' cleanup any leftover
  printf "Cleaning up previous rules ..."
  nRule = nRule + 1
  For iRec = nRule To 500
    sBuff = ruleName(iRec)
    Call removeRule(sBuff)
  Next

  ' check for errors
  If nErr > 0 Then
    printf "Error: " & nErr & " error(s) while loading rule(s)."
    Exit Function
  End If  
  
  ' all ok
  loadRules = True
End Function

' add a rule
Function addRule(sIPlist, nRule)
  Dim sName, NewRule

  On Error Resume Next  
  addRule = False
  
  ' setup rule name, remove previous (if existing)
  sName =  ruleName(nRule)
  printf "Creating rule " & Chr(34) & sName & chr(34) & " ..."
  If Not removeRule(sName) Then
    Exit Function
  End If
  
  ' create new rule object
  Set NewRule = CreateObject("HNetCfg.FWRule")
  If Err.Number <> 0 Then
    perror "Creating new fw rule object"
    Exit Function
  End If
  
  ' create new rule
  NewRule.Name = sName
  NewRule.Description = gsRuleDesc
  NewRule.Direction = NET_FW_RULE_DIR_IN
  NewRule.RemoteAddresses = sIPlist
  NewRule.Action = NET_FW_ACTION_BLOCK
  NewRule.Enabled = gbRuleEnabled
  NewRule.Profiles = goFwCurrProf
  goFwRules.Add NewRule
  If Err.Number <> 0 Then
    perror "Creating rule " & Chr(34) & sName & Chr(34)
    Exit Function
  End If
  
  ' all ok
  addRule = True
End Function

' remove a rule
Function removeRule(sRuleName)
  On Error Resume Next
  removeRule = False
  Err.Clear
  goFwPolicy.Rules.Remove sRuleName
  If Err.Number <> 0 Then
    perror "RemoveRule(" & sName & ")"
    Exit Function
  End If
  removeRule = True
End Function

' name for a rule
Function ruleName(nIndex)
  ruleName =  gsRulePrefix & Right("0000" & nIndex, 4)
End Function

' ===========================================================================
' SERVICE/COMMON
' ===========================================================================

' initializes script
Function initScript()
  Dim nPos
  
  On Error Resume Next

  gsBaseName = WScript.ScriptFullName
  nPos = InStrRev(gsBaseName, "\")
  If nPos > 0 Then
    gsBasePath = Mid(gsBaseName, 1, nPos)
    gsBaseName = Mid(gsBaseName, nPos + 1)
  End If
  nPos = InStrRev(gsBaseName, ".")
  If nPos > 0 Then
    gsBaseName = Mid(gsBaseName, 1, nPos - 1)
  End If
  
  gsLogFile = ""
  gsIniFile = gsBasePath & gsBaseName & ".ini"
  
  initScript = True
End Function

' loads the configuration options from file
Function loadConfig(sFile)
  Dim fp, iRec, vaRec
  Dim sRec, nPos
  Dim sKey, sVal
  
  On Error Resume Next
  loadConfig = False
  
  sRec = ""
  printf "Loading options from " & sFile & " ..."
  Set gdicConfig = CreateObject("Scripting.Dictionary")
  Set fp = fso.OpenTextFile(sFile, ForReading, False)
  If Not fp.AtEndOfStream Then
    sRec = fp.ReadAll
  End If
  fp.Close
  If Len(sRec)<1 Then
    printf "Missing or empty configuration file."
    Exit Function
  End If
  
  sRec = Replace(sRec, vbCrLf, vbLf)
  sRec = Replace(sRec, vbCr, vbLf)
  vaRec = Split(sRec, vbLf)
  
  For iRec = LBound(vaRec) To UBound(vaRec)
    sRec = Trim(vaRec(iRec))
    If Len(sRec)>0 Then
      If Mid(sRec,1,1)<>"#" And Mid(sRec,1,1)<>";" Then
        nPos = InStr(sRec, "=")
        If nPos > 0 Then
          sKey = UCase(Trim(Mid(sRec, 1, nPos - 1)))
          sVal = Trim(Mid(sRec, nPos + 1))
          gdicConfig.Add sKey, sVal
        End If
      End If    
    End If
  Next
  
  If gdicConfig.Count < 1 Then
    printf "Configuration file doesn't contain valid values."
    Exit Function
  End If
  loadConfig = True
End Function

' return a given config value
Function getConfValue(sKey, sDefault)
  Dim sKeyName, sValue
  
  On Error Resume Next
  sKeyName = UCase(sKey)
  If gdicConfig.Exists(sKeyName) Then
    sValue = gdicConfig.Item(sKeyName)
  Else
    sValue = sDefault    
  End If
  getConfValue = sValue
End Function

' output a message and write it to log
Sub printf(sTxt)
  logMsg sTxt
  WScript.StdOut.WriteLine sTxt
End Sub

' output an error (and write it to log)
Sub perror(sInf)
  Dim sMsg
  
  sMsg = "ERR: 0x" & Hex(Err.Number) & " " & Err.Description
  If Len(sInf) > 0 Then
    sMsg = sMsg & " :: " & sInf
  End If
  printf sMsg
End Sub

' write the log file
Sub logMsg(sTxt)
  Dim sLogName, dtFil
  Dim sRec, fp

  On Error Resume Next
  
  ' build the logname and handles rollover; the filename contains the
  ' month, so we'll have at max 12 files and the older ones (from prev
  ' year) will be deleted and overwritten as needed
  sLogName = gsBasePath & gsBaseName & Right("00" & Month(Now), 2) & ".log"
  If sLogName <> gsLogFile Then
    If fso.FileExists(sLogName) Then
      dtFil = fso.GetFile(sLogName).DateCreated
      If Year(dtFil) <> Year(Now) Then
        fso.DeleteFile sLogName
      End If
    End If
    gsLogFile = sLogName
  End If
  
  ' build the log record
  sRec = Right("0000" & Year(Now), 4) & "-" & _
         Right("00" & Month(Now), 2) & "-" & _
         Right("00" & Day(Now), 2) & " " & _
         Right("00" & Hour(Now), 2) & ":" & _
         Right("00" & Minute(Now), 2) & ":" & _
         Right("00" & Second(Now), 2) & " " & _
         sTxt

  ' write the logfile
  Set fp = fso.OpenTextFile(sLogName, ForAppending, True)
  fp.WriteLine sRec
  fp.Close
End Sub


